/*---------------------------------------------------------------------------*\

    FILE....: TPBX.CPP
    TYPE....: C++ Console program
    AUTHOR..: David Rowe
    DATE....: 31/05/02

    This program is a test program for the OpenSwitch12 that demos basic
    PBX functionality.

    Configuration:
    - 8 station/4 loop ports

    Incoming Calls:
    - when an incoming call "rings" a loop port, the first available
    station port is selected and starts ringing.  The ports are tried in
    the order 1,2,3,4,...,8
    - To pick up an incoming call on any other extension pick up the extension 
    and press "*".

    Outgoing Calls:
    - pick up a phone connected to a station port
    - press 0 to get an external line
    - the station will then be connected to the first available loop port.
    The loop ports are tried in the order 9,10,11,12.

    Internal Calls:
    - pick up a phone connected to a station port 
    - The extensions 1...8 map to the numbers 1...8, just dial the extension
    you want and it will start ringing.

    Transfers:
    - during a call generate a "hook flash" (e.g. press the flash or recall
    button on the phone, or tap the hook switch)
    - a dial tone will be produced
    - dial the number of the extension you wish to transfer to
    - it will start ringing
    - hangup to complete the transfer
    - if you wish to cancel the tranfer, generate a flash again
    - you can transfer an external line to another external line

    Hook Flash on external lines:
    - press # on the station phone

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <stdarg.h>

#include "../src/vpbapi.h"
#include "kbhit.h"

#define MAX_CH              12
#define STATIONS            8   // number of station ports

// states

#define LOOP_IDLE               0
#define LOOP_RING               1
#define LOOP_OFFHOOK            2
#define STAT_IDLE               3
#define STAT_LOOP_RING          4
#define STAT_OFFHOOK            5
#define STAT_DIAL               6
#define STAT_WAIT_OFFHOOK       7
#define STAT_WAIT_ONHOOK        8
#define STAT_WAIT_DIAL_FIN      9
#define STAT_WAIT_BUSY_FIN      10
#define STAT_WAIT_RINGBACK_FIN  11
#define STAT_WAIT_BUSY_FLSH     12
#define STAT_WAIT_DIAL_FINOH    13
#define STAT_WAIT_DIAL_OFFHK    14

// user defined vpb events, start at 200 to avoid confusion

#define MYEVT_LOOP_RING     200
#define MYEVT_STAT_ONHOOK   201
#define MYEVT_CALL_FINISHED 202
#define MYEVT_STAT_OFFHOOK  203
#define MYEVT_FLASH         204

#define RING_TIME_OUT  5000

// text translation for states

char *state_str[] = {
	"loop idle",
	"loop ring",
	"loop offhook",
	"stat idle",
	"stat loop ring",
	"stat offhook",
	"stat dial",
	"stat wait offhook",
	"stat wait onhook",
	"stat wait dial fin",
	"stat wait busy fin",
	"stat wait rngbk fin",
	"stat wait bsy flsh",
	"stat wait bsy fnoh",
	"stat wait dial offhk"
};

VPB_TONE DialTone     = {450, 425, 400, -10,   -10,  -10, 10000, 0   };
VPB_TONE BusyTone     = {425,   0,   0, -10,  -100, -100,   500,  500};
VPB_TONE RingBackTone = {425,   0,   0, -10,  -100, -100,  1000, 3000};
int syslog_level = 2;

// "ports" array: tells us which ports are being used
// == -1: free
// == -2: busy but not joined
// == x, where x=0..MAX_CH-1: joined with handle x

int ports[MAX_CH];

// records which port owns a parked port
int parked[MAX_CH];

void iterate_state_machine(int state[], void **timer, VPB_EVENT e, int dtmf[]);
int stat_dial_fin(int state[], VPB_EVENT e);
void translate_vpb_event(VPB_EVENT e, char s[]);
void mylog(int level, char *fmt, ...);
int flash_handler(int state, int h);
int loop_ringing(int state[]);
	
// "port" functions model aribration, allocation, and parking of ports

int port_get_free_loop(int station);
int port_get_free_station(int loop);
void port_busy_out(int h);
int port_get_joined(int h);
void port_free(int h);
int port_is_free(int h);
int port_join(int h1, int h2);
int port_get_num_busy_loops();
int port_get_num_busy_stations();
int port_is_joined(int h);

void port_park(int h);
void port_unpark(int h);
int port_has_parked(int h);
int port_get_parked(int h);
int port_am_i_parked(int h);
int port_get_parker(int h);

int main(int argc, char *argv[])
{
	int		handles[MAX_CH];
	VPB_EVENT	e;
	int	        ret,i;
	char            model_str[VPB_MAX_STR];
	int		state[MAX_CH];
	int             dtmf[MAX_CH];
	void            *timer[MAX_CH];

	vpb_seterrormode(VPB_DEVELOPMENT);

	for(i=0; i<MAX_CH; i++) {
		handles[i] = vpb_open(1,i+1);
		vpb_timer_open(&timer[i], handles[i], 0, RING_TIME_OUT);
		ports[i] = -1;
		parked[i] = -1;
	}
	vpb_get_model(model_str);
	printf("model = %s\n", model_str);

	// disable ring det on station ports
	for(i=0; i<STATIONS; i++) {
		state[i] = STAT_IDLE;
		vpb_disable_event(handles[i], VPB_MRING);
	}
	for(; i<MAX_CH; i++) {
		state[i] = LOOP_IDLE;
	}
	vpb_watchdog_enable(handles[0],1);

	do {
		ret = vpb_get_event_async(&e);
		if (ret == VPB_OK) {
			iterate_state_machine(state, timer, e, dtmf);
		}

		vpb_sleep(10);
		vpb_watchdog_reset(handles[0]);

	} while(!kbhit());

	for(i=0; i<MAX_CH; i++) {
		vpb_close(handles[i]);
	}

	return 0;
}

// async programming model guarantees only one event is being processed
// at any one time.  This is very useful in CT apps like this that use
// multiple ports whose activities must be coordinated.  Even though
// events may be generated simultaneously they will be processed
// sequentially.
//
// NOTE: I found this very difficult to write, and would appreciate anyone
// showimg me a better way to implement it.  With so many states, the 
// state machine starts to look like spaghetti code, with too many "gotos"
// - DR 27/8/02

/*
Tests:
- The hard part about writing this program was all the fail states
- The following details some of the test cases that caused problems during development

loop ring - loop stops ringing before station pickup
loop ring - stat ring - stat pick up - stat hangup
loop ring - stat ring - stat pick up - stat flash - stat hangup
loop ring - stat 0 busy - stat 1 should ring

stat pick up - stat hang up

stat pick up - press 0 - make external call - stat hangup
stat pick up - press 0 - no external lines free - stat hangup

stat A pick up - press 1-8 - stat B ring - dont pu stat B - stat A hangup
stat A pick up - press 1-8 - stat B ring - stat B pick up - stat A hangup - stat B hangup
stat A pick up - press 1-8 - stat B ring - stat B pick up - stat B hangup - stat A hangup
stat A pick up - press 1-8 - ext B busy - stat A hangup

stat A pick up - press 1-8 - stat B ring - stat B pu - stat B hangup - stat B pu - stat B dial 
stat A dial B - stat B pu - flash A - dial C - C pu - hangup B - C & A have call

loop ring - stat pick up - flash stat - hangup stat

*/

void iterate_state_machine(int state[], void **timer, VPB_EVENT e, int dtmf[]) {
	int             h;
	int             next_state;
	int             stat,statB;      // station handle for this call
	VPB_EVENT	estat;
	char		s[VPB_MAX_STR];

	h = e.handle;
	next_state = state[h];

	translate_vpb_event(e, s);
	
	switch(state[h]) {
			
	case LOOP_IDLE:
		if (e.type == VPB_RING) {
			// ringing on loop port

			port_busy_out(h);
			stat = port_get_free_station(h);
			if (stat != -1) {
				estat.handle = stat;
				estat.type = MYEVT_LOOP_RING;
				estat.data = 0;
				vpb_put_event(&estat);
				vpb_timer_start(timer[h]);
				next_state = LOOP_RING;	
			}
		}

		if (e.type == MYEVT_STAT_OFFHOOK) {
			// station port has told us to go off hook
			vpb_sethook_async(h, VPB_OFFHOOK);
			next_state = LOOP_OFFHOOK;
		}
		break;

	case LOOP_RING:
		// loop port is ringing

		if (e.type == VPB_RING) {
			vpb_timer_restart(timer[e.handle]);
		}
		if (e.type == VPB_TIMEREXP) {
			// time out since last ring, call finished
			estat.handle = port_get_joined(h);
			estat.type = MYEVT_CALL_FINISHED;
			estat.data = 0;
			vpb_put_event(&estat);
			port_free(h);
			next_state = LOOP_IDLE;
		}
		if (e.type == MYEVT_STAT_OFFHOOK) {
			// station picked up, so lets go offhook
			vpb_timer_stop(timer[h]);
			vpb_sethook_async(h, VPB_OFFHOOK);
			vpb_bridge(h,port_get_joined(h),VPB_BRIDGE_ON,0);
			next_state = LOOP_OFFHOOK;
		}
		break;
			
	case LOOP_OFFHOOK:
		// in loop-station call

		if (e.type == MYEVT_CALL_FINISHED) {
			// station on hook so hangup loop port
			vpb_sethook_async(h, VPB_ONHOOK);
			port_free(h);
			next_state = LOOP_IDLE;
		}
		if (e.type == MYEVT_FLASH) {
			vpb_dial_async(h, "&");
		}
		break;

	case STAT_IDLE:
		if (e.type == VPB_STATION_OFFHOOK) {
			port_busy_out(h);
			vpb_playtone_async(h, &DialTone);	
			next_state = STAT_DIAL;
		}
		if (e.type == MYEVT_LOOP_RING) {
			vpb_ring_station_sync(h, 1);
			next_state = STAT_LOOP_RING;
		}
		break;

	case STAT_LOOP_RING:
		// station is ringing due to incoming call

		if (e.type == MYEVT_CALL_FINISHED) {
			vpb_ring_station_sync(e.handle, 0);
			port_free(h);
			next_state = STAT_IDLE;
		}
				
		if (e.type == VPB_STATION_OFFHOOK) {
			// tell joined port we are off hook
			estat.handle = port_get_joined(h);
			estat.type =  MYEVT_STAT_OFFHOOK;
			estat.data = 0;
			vpb_put_event(&estat);
			next_state = STAT_OFFHOOK;
		}
		break;

	case STAT_DIAL:
		// dialtone presented to station, waiting for digit

		if (e.type == VPB_STATION_ONHOOK) {
			// hangup before digit pressed

			if (port_has_parked(h)) {
			        estat.handle = port_get_parked(h);
			        estat.type = MYEVT_CALL_FINISHED;
				estat.data = 0;
				vpb_put_event(&estat);
				port_unpark(h);				
			}
			vpb_tone_terminate(h);
			next_state = STAT_WAIT_DIAL_FINOH;
		}
			
		if (e.type == VPB_STATION_FLASH) {
			// unpark and rejoin
		
			if (port_has_parked(h)) {
				stat = port_get_parked(h);
				port_unpark(h);
				port_free(h);
				port_join(h, stat);
				vpb_bridge(h,stat,VPB_BRIDGE_ON,0);
				vpb_tone_terminate(h);
				next_state = STAT_WAIT_DIAL_OFFHK;
			}
		}
		
		next_state = STAT_OFFHOOK;		
		if (e.type == VPB_DTMF) {
			// digit pressed before dialtone ended

			vpb_tone_terminate(h);

			// remember digit as we need to wait until dial tone 
			// stops before processing further
			dtmf[h] = e.data;
			next_state = STAT_WAIT_DIAL_FIN;
		}

		if (e.type == VPB_DIALEND) {
			// dialtone ended before digit pressed, time out
			// with busy tone
			vpb_playtone_async(h, &BusyTone);
			next_state = STAT_WAIT_ONHOOK;
		}
		break;
		
	case STAT_WAIT_DIAL_FINOH:
		// wait for dial tone to stop, as caller has gone on hook

		if (e.type == VPB_DIALEND) {
			// is there a are parked port associated with this port
			if (port_has_parked(h)) {
				// tell parked port we have finished call
				estat.handle = port_get_parked(h);
				estat.type = MYEVT_CALL_FINISHED;
				estat.data = 0;
				vpb_put_event(&estat);
				
				port_unpark(h);
			}
			port_free(h);
			next_state = STAT_IDLE;
		}
		break;

	case STAT_WAIT_DIAL_FIN:
		// wait for dial tone to stop, then process digit

		if (e.type == VPB_DIALEND) {
			estat.handle = h;
			estat.type = VPB_DIGIT;
			estat.data = dtmf[h];
			next_state = stat_dial_fin(state, estat);
		}
		break;

	case STAT_WAIT_DIAL_OFFHK:
		// wait for dial tone to stop, then proceed to offhook state

		if (e.type == VPB_DIALEND) {
			next_state = STAT_OFFHOOK;
		}
		break;

	case STAT_WAIT_ONHOOK:
		// do nothing until station placed on hook

		if (e.type == VPB_STATION_ONHOOK) {
			if (port_has_parked(h)) {
				estat.handle = port_get_parked(h);
				estat.type = MYEVT_CALL_FINISHED;
				estat.data = 0;
				vpb_put_event(&estat);
				port_unpark(h);				
			}
			vpb_tone_terminate(h);
			next_state = STAT_WAIT_BUSY_FIN;
		}

		if (e.type == VPB_DIALEND) {
			// one busy tone iteration finished, play another
			vpb_playtone_async(h, &BusyTone);
		}

		if (e.type == VPB_STATION_FLASH) {
			// is there a are parked port associated with this port
			if (port_has_parked(h)) {
				// first stop busy tone
				vpb_tone_terminate(h);
				next_state = STAT_WAIT_BUSY_FLSH;
			}
		}
		break;
		
	case STAT_WAIT_BUSY_FLSH:
		// wait for busy tone to finish then unpark and reconnect

		if (e.type == VPB_DIALEND) {
			// unpark and reconnect
			stat = port_get_parked(h);
			port_unpark(h);
			port_join(h,stat);
			vpb_bridge(h,stat,VPB_BRIDGE_ON,0);
			next_state = STAT_OFFHOOK;
		}
		break;

	case STAT_WAIT_BUSY_FIN:
		// this state ensures we dont try to start dial tone before busy has stopped
		// playing.  This could happen if handset was picked up quickly after hangup
		if (e.type == VPB_DIALEND) {
			port_free(h);
			next_state = STAT_IDLE;
		}

		break;

	case STAT_WAIT_OFFHOOK:
		// wait for other station to go offhook, meanwhile generate ringback

		if (e.type == VPB_STATION_ONHOOK) {
			// calling station hang up before other stat answ
			vpb_tone_terminate(h);
			estat.handle = port_get_joined(h);
			estat.type = MYEVT_CALL_FINISHED;
			estat.data = 0;
			vpb_put_event(&estat);

			// check for parked port (esp loop port that needs to be told
			// to hang up)

			if (port_has_parked(h)) {
				estat.handle = port_get_joined(h);
				estat.type = MYEVT_CALL_FINISHED;
				estat.data = 0;
				vpb_put_event(&estat);
				port_unpark(h);				
			}
			next_state = STAT_WAIT_BUSY_FIN;
		}

		if (e.type == MYEVT_STAT_OFFHOOK) {
			// other stat went off hook
			vpb_tone_terminate(h);
			vpb_bridge(h,port_get_joined(h),VPB_BRIDGE_ON,0);
			next_state = STAT_WAIT_RINGBACK_FIN;
		}

		if (e.type == VPB_DIALEND) {
			// one ringback tone iteration finished, play another
			vpb_playtone_async(h, &RingBackTone);
		}

       		break;

	case STAT_WAIT_RINGBACK_FIN:
		// this state ensures ringback stops, just in case we try to quickly
		// flash and generate dial tone.

		if (e.type == VPB_DIALEND) {
			next_state = STAT_OFFHOOK;
		}

		break;

	case STAT_OFFHOOK:
		// state for station that is off hook

		if (e.type == MYEVT_CALL_FINISHED) {
			// other party has gone on hook
			
			vpb_playtone_async(h, &BusyTone);
			next_state = STAT_WAIT_ONHOOK;
		}

		if (e.type == VPB_STATION_FLASH) {
			next_state = flash_handler(state[h], h);
		}

		if ((e.type == VPB_DTMF) && (e.data == '#')) {
		        estat.handle = port_get_joined(h);
			estat.type = MYEVT_FLASH;
			estat.data = 0;
			vpb_put_event(&estat);
		}

		if (e.type == VPB_STATION_ONHOOK) {
			// on hook action depends on if we have a parked port or if we
			// are parked

			if (port_has_parked(h)) {
				// then connect far end port and parked port

				stat = port_get_parked(h);

				// kill this ports side of call
				statB = port_get_joined(h);
			        vpb_bridge(h,statB,VPB_BRIDGE_OFF,0);
				port_unpark(h);
				port_free(h);

				// join parked and far end
				port_join(stat,statB);
			        vpb_bridge(stat,statB,VPB_BRIDGE_ON,0);
				
				next_state = STAT_IDLE;
			} 
			else {
				if (port_am_i_parked(h)) {
					// then unpark
					stat = port_get_parker(h);
					port_unpark(stat);
					port_free(h);
					next_state = STAT_IDLE;
				}
				else {
					// hangup of regular call

				        // tell other end of call we are on hook so it
				        // can clean up it's side

					if (port_is_joined(h))  {
						vpb_bridge(h,port_get_joined(h),
							   VPB_BRIDGE_OFF,0);
						estat.handle = port_get_joined(h);
						estat.type = MYEVT_CALL_FINISHED;
						estat.data = 0;
						vpb_put_event(&estat);
					}
					port_free(h);
					next_state = STAT_IDLE;
				}
			}
		}

		break;

	default:
		assert(0);
	}
			       
	translate_vpb_event(e, s);
	
	mylog(1, "%s ST: %20s  NST: %20s bl: %02d bs: %02d\n", s, state_str[state[h]],
	      state_str[next_state], port_get_num_busy_loops(), 
	      port_get_num_busy_stations());
	
	state[h] = next_state;
}

int flash_handler(int state, int h) {
	int next_state = state;

	if (port_has_parked(h)) {
		// port already associated with parked port - do nothing
	}
	else {
		// port not associated with parked port, so set up new
		// station to station call with this 
		// station as A-party
		vpb_bridge(h,port_get_joined(h),VPB_BRIDGE_OFF,0);
		port_park(h);
		vpb_playtone_async(h, &DialTone);	
		next_state = STAT_DIAL;		
	}

	return next_state;
}

// state handler for STAT_DIAL state, broken out into function due to it's
// size.  Handles dialling.

int stat_dial_fin(int states[], VPB_EVENT e) {
	int       h = e.handle, state;
	int       next_state;
	int       stat;
	int       loop;
	VPB_EVENT estat;

	state = states[h];
	next_state = state;

	if (e.data == '0') {
		// get external line
	        // note: dont allow external line oif we have a parked port, as this would
	        // be bridging two external lines, where it is difficult to detect hangup.
		if (port_has_parked(h))
		  loop = -1;
		else
		  loop = port_get_free_loop(h);
		if (loop != -1) {
			estat.handle = loop;
			estat.type = MYEVT_STAT_OFFHOOK;
			estat.data = 0;
			vpb_put_event(&estat);
			next_state = STAT_OFFHOOK;
			vpb_bridge(h,port_get_joined(h),VPB_BRIDGE_ON,0);
		}
		else {
			// no lines available
			vpb_playtone_async(h, &BusyTone);
			next_state = STAT_WAIT_ONHOOK;
		}
	} 
	else {
		if (e.data == '*') {
			// pick up incoming call that is ringing another station 
			// (only if from a loop port)
			loop = loop_ringing(states);
			printf("* loop = %d\n", loop);
			if (loop != -1) {
				// tell ringing station to stop ringing
				stat = port_get_joined(loop);
				estat.handle = stat;
				estat.type = MYEVT_CALL_FINISHED;
			        estat.data = 0;
				vpb_put_event(&estat);
				
				// connect this station to ringing loop port

				port_free(loop);
				port_join(h, loop);
				estat.handle = loop;
				estat.type =  MYEVT_STAT_OFFHOOK;
				estat.data = 0;
				vpb_put_event(&estat);
				next_state = STAT_OFFHOOK;
				
			}
			else {
				// no incoming calls
				vpb_playtone_async(h, &BusyTone);
				next_state = STAT_WAIT_ONHOOK;
			}
				
		}		
			
		else {
			// internal call to station
			if ((e.data >= '1') && (e.data <= '8')) {
				stat = e.data - '1';
				if (port_is_free(stat)) {
					port_join(h, stat);
					estat.handle = stat;
					estat.type = MYEVT_LOOP_RING;
					estat.data = 0;
					vpb_put_event(&estat);
					next_state = STAT_WAIT_OFFHOOK;
					vpb_playtone_async(h, &RingBackTone);
				}
				else {
					// station is busy
					vpb_playtone_async(h, &BusyTone);
					next_state = STAT_WAIT_ONHOOK;
				}
			}
			else {
				// invalid station number
				vpb_playtone_async(h, &BusyTone);
				next_state = STAT_WAIT_ONHOOK;
			}
					
		}
				
	}
	return next_state;
}

// return handle of first loop port in ringing state, -1 if none ringing

int loop_ringing(int state[]) {
	int i;

	for(i=STATIONS; i<MAX_CH; i++)
		if (state[i] == LOOP_RING) {
			return i;
		}
			
	return -1;
}


// look for a free station port, and "joins" it to the supplied loop port
// returns -1 if no stations free

int port_get_free_station(int loop) {
	int i;

	assert(ports[loop] != -1);

	for(i=0; i<STATIONS; i++)
		if (ports[i] == -1) {
			ports[i] = loop;
			ports[loop] = i;
			return i;
		}
			
	mylog(2, "port: loop %d was allocated station %d\n", loop, ports[loop]);
	return -1;
}

// look for a free loop port, and "joins it" to the supplied station port
// returns -1 if no loop ports free

int port_get_free_loop(int station) {
	int i;

	assert(ports[station] != -1);

	for(i=STATIONS; i<MAX_CH; i++)
		if (ports[i] == -1) {
			ports[i] = station;
			ports[station] = i;
			return i;
		}
			
	mylog(2, "port: station %d was allocated loop %d\n", station, 
	      ports[station]);
	return -1;
}

// frees port with handle h
void port_free(int h) {
	assert(ports[h] != -1);

	int joined = ports[h];

	if (joined != -2) {
		// if joined unjoin, mark other end of join as busy
		ports[h] = -1;
		ports[joined] = -2;
		mylog(2, "port: %d freed and joined port %d busy\n", h, joined);
	}
	else {
		// just one port busy
		ports[h] = -1;
		mylog(2, "port: %d freed\n", h);
	}
}

// returns handle of port joined to this one.  Should only be called for 
// joined ports
int port_get_joined(int h) {
	assert(ports[h] != -1);
	assert(ports[h] != -2);
	assert(ports[h] != -3);
	return ports[h];
}

// "busies out" a port
void port_busy_out(int h) {
	assert(ports[h] == -1);
	ports[h] = -2;
	mylog(2, "port: %d busied out\n",h);
}

// returns 1 if a port is free, 0 if port is busy
int port_is_free(int h) {
	if (ports[h] == -1)
		return 1;
	else
		return 0;
}

// join two ports

int port_join(int h1, int h2) {

	// make sure not joined already, but OK to be busy or free
	assert(ports[h1] < 0);
	assert(ports[h2] < 0);

	ports[h1] = h2;
	ports[h2] = h1;

	mylog(2, "port: %d joined to %d busied out\n", h1, h2);
	return -1;
}

// returns 1 if port is joined, 0 otherwise

int port_is_joined(int h) {
  
        if (ports[h] >=0)
		return 1;
	else
		return 0;
}

int port_get_num_busy_loops() {
	int i, busy;

	busy = 0;
	for(i=STATIONS; i<MAX_CH; i++)
		if (ports[i] != -1) 
			busy++;

	return busy;
}

int port_get_num_busy_stations() {
	int i, busy;

	busy = 0;
	for(i=0; i<STATIONS; i++)
		if (ports[i] != -1) 
			busy++;

	return busy;
}

// Park the far-end port joined to port h.  Returns 1 if OK, 0 if
// port already parked

void port_park(int h) {
	assert(ports[h] >= 0);
	assert(parked[h] == -1);

	parked[h] = ports[h];  // record far end port
	ports[h] = -2;         // flag near end as busy 
	ports[parked[h]] = -2; // flag far end as busy

	mylog(2, "port: %d just parked port %d\n", h, parked[h]);
}

// Unpark

void port_unpark(int h) {
	assert(parked[h] != -1);
	int pkd = parked[h];

	parked[h] = -1;

	mylog(2, "port: %d just unparked port pkd\n", h, pkd);
}

// returns 1 if there is a parked port associated with this port

int port_has_parked(int h) {
	if (parked[h] != -1)
		return 1;
	else
		return 0;
}

// returns 1 if port h is parked

int port_am_i_parked(int h) {
	int i;

	for(i=0; i<MAX_CH; i++)
		if (parked[i] == h)
			return 1;
	      
	return 0;
}

// returns port that parked port h

int port_get_parker(int h) {
	int i;

	for(i=0; i<MAX_CH; i++)
		if (parked[i] == h)
			return i;
	      
	return 0;
}

// returns parked port associated with port h
int port_get_parked(int h) {
	assert(parked[h] != -1);
	return parked[h];
}

void translate_vpb_event(VPB_EVENT e, char s[]) {

	switch(e.type) {
	case VPB_RING:
		sprintf(s, "[%02d] VPB_RING           ", e.handle); 
		break;			
	case VPB_DIGIT:		
		sprintf(s, "[%02d] VPB_DIGIT", e.handle); 
		break;
	case VPB_TONEDETECT:		
		sprintf(s, "[%02d] VPB_TONEDETECT %d",e.handle, e.data); 
		break;
	case VPB_TIMEREXP:		
		sprintf(s, "[%02d] VPB_TIMEREXP       ", e.handle); 
		break;
	case VPB_VOXON:		
		sprintf(s, "[%02d] VPB_VOXON", e.handle); 
		break;
	case VPB_VOXOFF:		
		sprintf(s, "[%02d] VPB_VOXOFF", e.handle); 
		break;
	case VPB_DTMF:		
		sprintf(s, "[%02d] VPB_DTMF [%c]       ", e.handle, 
			(char)e.data); 
		break;
	case VPB_STATION_OFFHOOK:	
		sprintf(s, "[%02d] VPB_STATION_OFFHOOK", e.handle); 
		break;
	case VPB_STATION_ONHOOK:	
		sprintf(s, "[%02d] VPB_STATION_ONHOOK ", e.handle); 
		break;
	case VPB_RING_OFF:		
		sprintf(s, "[%02d] VPB_RING_OFF       ", e.handle); 
		break;
	case VPB_DROP:		
		sprintf(s, "[%02d] VPB_DROP", e.handle); 
		break;
	case VPB_STATION_FLASH:
		sprintf(s, "[%02d] VPB_STATION_FLASH", e.handle); 
		break;
	case VPB_PLAYEND:	
		sprintf(s, "[%02d] VPB_PLAYEND", e.handle); 
		break;
	case VPB_RECORDEND:	
		sprintf(s, "[%02d] VPB_RECORDEND", e.handle); 
		break;
	case VPB_DIALEND:
		sprintf(s, "[%02d] VPB_DIALEND        ", e.handle); 
		break;

	// user-defined events

	case MYEVT_LOOP_RING:
		sprintf(s, "[%02d] MYEVTLOOP_RING     ", e.handle); 
		break;
	case MYEVT_STAT_ONHOOK:
		sprintf(s, "[%02d] MYEVT_STAT_ONHOOK  ", e.handle); 
		break;
	case MYEVT_CALL_FINISHED:
		sprintf(s, "[%02d] MYEVT_CALL_FINISHED", e.handle); 
		break;
	case MYEVT_STAT_OFFHOOK:
		sprintf(s, "[%02d] MYEVT_STAT_OFFHOOK ", e.handle); 
		break;
	case MYEVT_FLASH:
		sprintf(s, "[%02d] MYEVT_FLASH        ", e.handle); 
		break;
	default:
		sprintf(s, "[%02d] [%c] Che? %d ", e.handle, (char)e.data,
			e.type); 
		break;
	}

}

/*--------------------------------------------------------------------------*\

	FUNCTION....: mylog
	AUTHOR......: David Rowe
	DATE CREATED: 10/10/01

	Allows switching between syslog and console output of messages.

\*--------------------------------------------------------------------------*/

void mylog(int level, char *fmt, ...) {
  char    s[VPB_MAX_STR];
  va_list argptr;

  va_start(argptr, fmt);
  vsprintf(s, fmt, argptr);
  va_end(argptr);
  
  if(syslog_level >= level) {
    // syslog(LOG_INFO, s);
    printf("%s",s);
  }

}



